package com.ejie.p06b.dao;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.Properties;

import javax.naming.NamingException;

import com.ejie.p06b.database.P06bConnector;
import com.ejie.p06b.log.P06BLog;


public class P06bDao {
	
	protected Connection con = null;
	protected Statement stmt = null;
	protected PreparedStatement pstmt = null;
	protected String sql = null;
	protected ResultSet rs = null;
	protected boolean autoCommit = true;
	
	/**
	 * Constructor por defecto
	 */
	public P06bDao() {
		super();
	}
	
	/**
	 * Constructor completo
	 * 
	 * @param con
	 *            : Objeto de tipo Connection
	 * @param stmt
	 *            : Objeto de tipo Statement
	 * @param pstmt
	 *            : Objeto de tipo PreparedStatement
	 * @param sql
	 *            : Objeto de tipo String con la consulta SQL a ejecutar
	 * @param rs
	 *            : Objeto de tipo ResultSet
	 * @param autoCommit
	 *            : Objeto de tipo boolean que indica si se debe hacer un commit
	 *            automtico
	 */
	public P06bDao(Connection con, Statement stmt, PreparedStatement pstmt,
			String sql, ResultSet rs, boolean autoCommit) {
		super();
		this.con = con;
		this.stmt = stmt;
		this.pstmt = pstmt;
		this.sql = sql;
		this.rs = rs;
		this.autoCommit = autoCommit;
	}

	/**
	 * Constructor en base slo a un objeto Connection
	 * 
	 * @param con
	 *            : Objeto de tipo Connection
	 */
	public P06bDao(Connection con) {
		super();
		this.con = con;
		this.stmt = null;
		this.pstmt = null;
		this.sql = null;
		this.rs = null;
		this.autoCommit = false;
	}
	
	/**
	 * Mtodo de conexion. Sirve para llamar dinmicamente a mtodos del DAO
	 * creando conexion nueva independiente.
	 * 
	 * @param parametros
	 *            : Objeto de tipo Properties con los parmetros de conexin a
	 *            base de datos
	 * @throws Exception
	 *             : Lanza un objeto de tipo Exception en caso de haber existido
	 *             algn problema
	 */
	public void conectar(Properties parametros) throws NamingException, SQLException {
		this.con = P06bConnector.openConnection(parametros);
		this.con.setAutoCommit(this.autoCommit);
		P06BLog.send(0,this, "(conectar) Conexin a base de datos abierta");
	}

	/**
	 * Mtodo de conexion. Sirve para llamar dinmicamente a mtodos del DAO
	 * creando conexion nueva independiente.
	 * 
	 * @param parametros
	 *            : Objeto de tipo Properties con los parmetros de conexin a
	 *            base de datos
	 * @param autocommit
	 *            : Objeto de tipo boolean que indica si se debe hacer un commit
	 *            automtico
	 * @throws Exception
	 *             : Lanza un objeto de tipo Exception en caso de haber existido
	 *             algn problema
	 */
	public void conectar(Properties parametros, boolean autocommit)
			throws Exception {
		this.con = P06bConnector.openConnection(parametros);
		this.autoCommit = autocommit;
		this.con.setAutoCommit(this.autoCommit);
		P06BLog.send(0,this, "(conectar) Conexin a base de datos abierta");
	}

	/**
	 * Mtodo de desconexin. Sirve para desconectar la conexin dinmica
	 * anteriormente creada.
	 * 
	 * @throws SQLException
	 *             : Lanza un objeto de tipo SQLException en caso de haber
	 *             existido algn problema
	 */
	public void desconectar() throws SQLException {

		if (this.autoCommit == false && !this.con.isClosed()) {
			this.stmt = this.con.createStatement();
			this.stmt.executeUpdate("COMMIT");
			this.con.setAutoCommit(this.autoCommit);
			this.stmt.close();
			P06BLog.send(0,this, "(desconectar) Commit de base de datos realizado");
		}

//		try {
			if (this.rs != null) {
				this.rs.close();
			}
			if (this.stmt != null) {
				this.stmt.close();
			}
			if (this.pstmt != null) {
				this.pstmt.close();
			}
			if (this.con != null && !this.con.isClosed()) {
				this.con.close();
			}
//			P06BLog.send(0,this, "(desconectar) Conexin a base de datos cerrada");
//		} catch (Exception excepcion) {
//			P06BLog.send(0,this, "(desconectar) Error al desconectarse de base de datos");
//		}

	}

	/**
	 * Mtodo de rollback. Vuelve al estado anterior de base de datos,
	 * descartando los cambios sin confirmacin
	 */
	public void rollback() {
		try {
			if (this.con != null && !this.con.isClosed()) {
				this.con.rollback();
			}
			P06BLog.send(0,this, "(rollback) Rollback de base de datos realizado");
		} catch (Exception excepcion) {
			P06BLog.send(0,this, "(rollback) Error al realizar el rollback de base de datos");
		}
	}

	/**
	 */
	public void fijarNumericCharacters() throws SQLException {
		this.stmt = this.con.createStatement();
		this.stmt.executeUpdate("alter session set NLS_NUMERIC_CHARACTERS = ',.'");
		this.stmt.close();
		P06BLog.send(0,this, "alter session set NLS_NUMERIC_CHARACTERS = ',.' Ok");
	}

	/**
	 * Mtodo de rollback. Confirma los cambios realizados en base de datos
	 * 
	 * @throws SQLException
	 *             : Lanza un objeto de tipo SQLException en caso de haber
	 *             existido algn problema
	 */
	public void commit() throws SQLException {
		if (this.con != null && !this.con.isClosed()) {
			this.stmt = this.con.createStatement();
			this.stmt.executeUpdate("COMMIT");
			this.con.setAutoCommit(this.autoCommit);
			this.stmt.close();
			P06BLog.send(0,this, "(commit) Commit de base de datos realizado");
		}
	}

	/**
	 * Mtodo que comprueba si la conexin de base de datos ya est cerrada
	 * 
	 * @return Devuelve un objeto boolean que indica si la conexin a base de
	 *         datos est cerrada (true) o no (false)
	 * @throws SQLException
	 *             : Lanza un objeto de tipo SQLException en caso de haber
	 *             existido algn problema
	 */
	public boolean isClosed() throws SQLException {
		return (this.con != null ? this.con.isClosed() : true);
	}

	/**
	 * Mtodo que obtiene la propiedad de confirmacin de cambios en base de
	 * datos de manera automtica
	 * 
	 * @return Devuelve un objeto boolean que indica si la confirmacin
	 *         automtica de cambios en base de datos est activa (true) o no
	 *         (false)
	 */
	public boolean isAutoCommit() {
		return this.autoCommit;
	}

	/**
	 * Mtodo que define la propiedad de confirmacin de cambios en base de
	 * datos de manera automtica
	 * 
	 * @param autoCommit
	 *            : Objeto de tipo boolean que indica si los cambios en base de
	 *            datos se quiere que se confirmen automticamente o no
	 * @throws SQLException
	 *             : Lanza un objeto de tipo SQLException en caso de haber
	 *             existido algn problema
	 */
	public void setAutoCommit(boolean autoCommit) throws SQLException {
		this.autoCommit = autoCommit;
		if (this.con != null && !this.con.isClosed()) {
			this.con.setAutoCommit(this.autoCommit);
		}
	}

	/**
	 * Mtodo que obtiene el objeto de conexin previamente creado
	 * 
	 * @return Devuelve un objeto Connection con la conexin a base de datos
	 *         previamente creada
	 */
	public Connection getCon() {
		return this.con;
	}

	/**
	 * Mtodo que define el objeto de conexin
	 * 
	 * @param con
	 *            : Objeto de tipo Connection con la conexin a base de datos
	 */
	public void setCon(Connection con) {
		this.con = con;
	}

}
